/*
 * Decompiled with CFR 0.152.
 */
package certa.vics.tools;

import certa.modbus.client.RtuTransportJSerialComm;
import certa.vics.modbus.FlashModbusClient;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryTool {
    private static final Logger log = LoggerFactory.getLogger(MemoryTool.class);
    static final int MEM_FLASH = 0;
    static final int MEM_EEPROM = 1;
    static final int MEM_NVRAM = 2;
    static final int MEM_EXT_EEPROM = 3;
    static final String[] MEM_TYPES = new String[]{"FLASH", "EEPROM", "NVRAM", "Ext.EEPROM"};
    final String comPort;
    final int speed;
    final int id;
    final int password;
    private static final int CHUNK_SIZE = 248;

    public MemoryTool(String comPort, int speed, int id, int password) {
        this.comPort = comPort;
        this.speed = speed;
        this.id = id;
        this.password = password;
    }

    public void setWriteMode(FlashModbusClient mc, int id, int password, boolean writeMode) throws Exception {
        block5: {
            int attempt = 0;
            if (mc != null) {
                log.info("Setting write mode to {}", (Object)writeMode);
                while (true) {
                    boolean exception;
                    mc.InitSetWriteModeRequest(id, password, writeMode);
                    if (mc.execRequest()) {
                        int model = mc.readInt16FromPDU(4, true);
                        int fw = mc.readInt16FromPDU(6, true);
                        System.out.println("OK. Model: " + model + ", firmware: " + fw);
                        break block5;
                    }
                    boolean bl = exception = mc.getResult() == 2;
                    if (exception && mc.getExceptionCode() == 16) {
                        throw new Exception("Wrong password (error 16)");
                    }
                    if (!exception || mc.getExceptionCode() != 17) break;
                    if (++attempt > 10) {
                        throw new Exception("Timeout while entering write mode");
                    }
                    log.warn("Need more time (error 17)");
                    Thread.sleep(500L);
                }
                throw new Exception(mc.getResultAsString());
            }
        }
    }

    private int chunksCount(int bytesCount) {
        int i = bytesCount / 248;
        if (bytesCount % 248 > 0) {
            ++i;
        }
        return i;
    }

    private void readChunk(FlashModbusClient mc, int id, int memType, int chunk, byte[] buf, int memStart) throws Exception {
        int offset = chunk * 248;
        int size = Math.min(248, buf.length - offset);
        int memAddr = memStart + offset;
        System.out.println("Read part " + (chunk + 1) + " at " + memAddr + ", " + size + " bytes");
        if (memType == 0) {
            mc.InitReadFlashRequest(id, memAddr, size);
        } else if (memType == 1) {
            mc.InitReadIntEepromRequest(id, memAddr, size);
        } else if (memType == 2) {
            mc.InitReadNvramRequest(id, memAddr, size);
        } else if (memType == 3) {
            mc.InitReadExtEepromRequest(id, memAddr, size);
        } else {
            throw new IllegalStateException("MemType=" + memType);
        }
        if (!mc.execRequest()) {
            throw new Exception(mc.getResultAsString());
        }
        mc.readFromPdu(4, size, buf, offset);
    }

    private void read(FlashModbusClient mc, int id, int memType, byte[] buf, int memStart) throws Exception {
        for (int i = 0; i < this.chunksCount(buf.length); ++i) {
            this.readChunk(mc, id, memType, i, buf, memStart);
        }
        System.out.println("OK");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void saveBinFile(byte[] data, String fileName) throws IOException {
        try (FileOutputStream stream = new FileOutputStream(fileName);){
            stream.write(data);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void readMemoryToFile(int memType, int addr, int size, String fileName) throws Exception {
        System.out.println();
        System.out.println("Read " + MEM_TYPES[memType] + " at address " + addr + ", size " + size + " (" + this.comPort + ", " + this.speed + ", id:" + this.id + ", password=" + this.password + ").");
        System.out.println("Press Ctrl+C to break.");
        System.out.println();
        byte[] data = new byte[size];
        boolean ok = false;
        try (FlashModbusClient mc = new FlashModbusClient();){
            mc.setTransport(new RtuTransportJSerialComm(this.comPort, this.speed, 8, 0, 2, 1000, 100));
            System.out.println("Reading " + MEM_TYPES[memType] + "...");
            this.read(mc, this.id, memType, data, addr);
            ok = true;
        }
        if (ok) {
            System.out.println("Saving file " + fileName);
            this.saveBinFile(data, fileName);
            System.out.println("DONE!");
        }
    }

    private byte[] preparePage(int offset, byte[] data) {
        byte[] page = new byte[256];
        Arrays.fill(page, (byte)-1);
        for (int i = 0; i < 256 && i + offset < data.length; ++i) {
            page[i] = data[i + offset];
        }
        return page;
    }

    private void writeFlashPage(FlashModbusClient mc, int id, int pageNum, byte[] data) throws Exception {
        if (mc != null) {
            System.out.print("Writing page: " + pageNum + " ... ");
            System.out.println("OK");
            log.info("Send part 1 of page {}", (Object)pageNum);
            mc.InitStoreFlashHalfPageRequest(id, pageNum, data, false);
            if (!mc.execRequest()) {
                throw new Exception(mc.getResultAsString());
            }
            log.info("Send part 2 of page {}", (Object)pageNum);
            mc.InitStoreFlashHalfPageRequest(id, pageNum, data, true);
            if (!mc.execRequest()) {
                throw new Exception(mc.getResultAsString());
            }
            log.info("Write page {} to flash", (Object)pageNum);
            mc.InitWriteFlashPageRequest(id, pageNum);
            if (!mc.execRequest()) {
                throw new Exception(mc.getResultAsString());
            }
        }
    }

    public void writeFlash(FlashModbusClient mc, int id, int address, byte[] data) throws Exception {
        if (address > 127) {
            throw new Exception("FLASH write address is a page number (0..127)");
        }
        int cnt = data.length / 256;
        if (data.length % 256 > 0) {
            ++cnt;
        }
        System.out.println("First page: " + address + ", pages count: " + cnt);
        for (int i = 0; i < cnt; ++i) {
            this.writeFlashPage(mc, id, address + i, this.preparePage(i * 256, data));
        }
    }

    private void writeChunk(FlashModbusClient mc, int id, byte[] data, int chunk, int startAddr, int memType, int dataSize) throws Exception {
        if (mc != null) {
            int addr = chunk * 248;
            byte[] cdata = Arrays.copyOfRange(data, addr, addr + Math.min(248, dataSize - addr));
            addr += startAddr;
            if (memType == 2) {
                log.info("Write {} bytes to NVRAM at {} (chunk {})", cdata.length, addr, chunk + 1);
                mc.InitWriteNvramRequest(id, addr, cdata);
            } else if (memType == 3) {
                log.info("Write {} bytes to External EEPROM at {} (chunk {})", cdata.length, addr, chunk + 1);
                mc.InitWriteExtEepromRequest(id, addr, cdata);
            } else {
                log.info("Write {} bytes to Internal EEPROM at {} (chunk {})", cdata.length, addr, chunk + 1);
                mc.InitWriteIntEepromRequest(id, addr, cdata);
            }
            if (!mc.execRequest()) {
                throw new Exception(mc.getResultAsString());
            }
        }
    }

    public void writeNonFlash(FlashModbusClient mc, int memType, int id, int address, byte[] data) throws Exception {
        for (int i = 0; i < this.chunksCount(data.length); ++i) {
            this.writeChunk(mc, id, data, i, address, memType, data.length);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void writeMemoryFromFile(File file, int memType, int addr, int size) throws Exception {
        System.out.println();
        System.out.println("Write " + MEM_TYPES[memType] + " at address " + addr + ", size " + size + " (" + this.comPort + ", " + this.speed + ", id:" + this.id + ", password=" + this.password + ").");
        System.out.println("Press Ctrl+C to break.");
        System.out.println();
        byte[] data = new byte[size];
        int res = 0;
        try (FileInputStream inputStream = new FileInputStream(file);){
            res = ((InputStream)inputStream).read(data);
        }
        System.out.println(res + " bytes read from file " + file.getAbsolutePath());
        System.out.println();
        boolean ok = false;
        try (FlashModbusClient mc = new FlashModbusClient();){
            mc.setTransport(new RtuTransportJSerialComm(this.comPort, this.speed, 8, 0, 2, 3000, 100));
            if (memType == 0) {
                System.out.println("Enter programming mode...");
                this.setWriteMode(mc, this.id, this.password, true);
            }
            System.out.println("Writing " + MEM_TYPES[memType] + "...");
            if (memType == 0) {
                this.writeFlash(mc, this.id, addr, data);
            } else if (memType == 1 || memType == 2 || memType == 3) {
                this.writeNonFlash(mc, memType, this.id, addr, data);
            } else {
                throw new IllegalStateException("MemType=" + memType);
            }
            if (memType == 0) {
                System.out.println("Exit programming mode...");
                this.setWriteMode(mc, this.id, this.password, false);
            }
            ok = true;
        }
        if (ok) {
            System.out.println("DONE!");
        }
    }

    public static void main(String[] args) {
        try {
            if (args.length < 7) {
                System.out.println("ViCS Memory Tool");
                System.out.println("Command line parameters: <COM_port> <speed> <id> <password> -<command> <address> <size> <input_file>");
                System.out.println("  commands:");
                System.out.println("    -ReadFlash");
                System.out.println("    -WriteFlash   (WARNING: address is a page number from 0 to 127)");
                System.out.println("    -ReadEEPROM");
                System.out.println("    -WriteEEPROM");
                System.out.println("    -ReadNVRAM");
                System.out.println("    -WriteNVRAM");
                System.out.println("    -ReadExt");
                System.out.println("    -WriteExt");
                return;
            }
            String cmd = args[4];
            int addr = Integer.parseInt(args[5]);
            int size = Integer.parseInt(args[6]);
            File inFile = null;
            if (cmd != null && cmd.contains("Write")) {
                if (args.length < 8) {
                    throw new Exception("Input file not specified");
                }
                inFile = new File(args[7]);
                if (!inFile.exists()) {
                    throw new Exception("File does not exist: " + inFile.getAbsolutePath());
                }
            }
            MemoryTool tool = new MemoryTool(args[0], Integer.parseInt(args[1]), Integer.parseInt(args[2]), Integer.parseInt(args[3]));
            if ("-ReadFlash".equalsIgnoreCase(cmd)) {
                tool.readMemoryToFile(0, addr, size, "_flash_" + addr + "_" + size + ".bin");
            } else if ("-ReadEEPROM".equalsIgnoreCase(cmd)) {
                tool.readMemoryToFile(1, addr, size, "_eeprom_" + addr + "_" + size + ".bin");
            } else if ("-ReadNVRAM".equalsIgnoreCase(cmd)) {
                tool.readMemoryToFile(2, addr, size, "_nvram_" + addr + "_" + size + ".bin");
            } else if ("-ReadExt".equalsIgnoreCase(cmd)) {
                tool.readMemoryToFile(3, addr, size, "_external_" + addr + "_" + size + ".bin");
            } else if ("-WriteFlash".equalsIgnoreCase(cmd)) {
                tool.writeMemoryFromFile(inFile, 0, addr, size);
            } else if ("-WriteEEPROM".equalsIgnoreCase(cmd)) {
                tool.writeMemoryFromFile(inFile, 1, addr, size);
            } else if ("-WriteNVRAM".equalsIgnoreCase(cmd)) {
                tool.writeMemoryFromFile(inFile, 2, addr, size);
            } else if ("-WriteExt".equalsIgnoreCase(cmd)) {
                tool.writeMemoryFromFile(inFile, 3, addr, size);
            } else {
                System.out.println("UNKNOWN COMMAND : " + cmd);
            }
        }
        catch (NoSuchFileException e) {
            System.out.println("");
            System.out.println("ERROR. File not found: " + e.getMessage());
        }
        catch (Throwable e) {
            System.out.println("");
            System.out.println("ERROR (" + e.getClass().getName() + "): " + e.getMessage());
            System.out.println("");
            if (System.getProperty("debug") != null) {
                e.printStackTrace();
            }
            log.error("EXCEPTION", e);
        }
    }

    public static interface ProgressListener {
        public void updateStep(int var1);
    }
}

